@extends('layouts.app')

@section('content')
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
<script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>

<style>
    .stat-card {
        border-radius: 0.5rem;
        transition: transform 0.2s ease, box-shadow 0.2s ease;
        min-height: 120px;
    }
    .stat-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    }
</style>

<div class="content">
    <div class="container-fluid py-4">
        <!-- Header -->
        <div class="d-flex justify-content-between align-items-center mb-4">
            <div>
                <h1 class="h3 mb-0">Accounting</h1>
                <p class="text-secondary mb-0">Financial overview, profit & loss, and ledger.</p>
            </div>
        </div>

        <!-- Date Filter -->
        <div class="card shadow-sm mb-4">
            <div class="card-body">
                <form action="{{ url('accounting') }}" method="GET" class="row g-3 align-items-end">
                    <div class="col-12 col-md-4">
                        <label for="start_date" class="form-label">Start Date</label>
                        <input type="text" id="start_date" name="start_date" class="form-control" 
                            value="{{ request('start_date', '') }}" placeholder="Select start date (optional)" autocomplete="off">
                    </div>
                    <div class="col-12 col-md-4">
                        <label for="end_date" class="form-label">End Date</label>
                        <input type="text" id="end_date" name="end_date" class="form-control" 
                            value="{{ request('end_date', '') }}" placeholder="Select end date (optional)" autocomplete="off">
                    </div>
                    <div class="col-12 col-md-4">
                        <button type="submit" class="btn btn-primary w-100 mb-2">
                            <i class="bi bi-filter me-2"></i>Apply Filter
                        </button>
                        @if(request('start_date') || request('end_date'))
                        <a href="{{ url('accounting') }}" class="btn btn-outline-secondary w-100">
                            <i class="bi bi-x-circle me-2"></i>Clear Filter
                        </a>
                        @endif
                    </div>
                </form>
                @if(request('start_date') || request('end_date'))
                <div class="mt-3">
                    <small class="text-muted">
                        <i class="bi bi-info-circle me-1"></i>
                        Showing data from <strong>{{ \Carbon\Carbon::parse($startDate)->format('d M Y') }}</strong> 
                        to <strong>{{ \Carbon\Carbon::parse($endDate)->format('d M Y') }}</strong>
                    </small>
                </div>
                @else
                <div class="mt-3">
                    <small class="text-muted">
                        <i class="bi bi-info-circle me-1"></i>
                        Showing all data from <strong>{{ \Carbon\Carbon::parse($startDate)->format('d M Y') }}</strong> 
                        to <strong>{{ \Carbon\Carbon::parse($endDate)->format('d M Y') }}</strong>
                    </small>
                </div>
                @endif
            </div>
        </div>

        <!-- Summary Cards -->
        <div class="row g-3 mb-4">
            <!-- Card 1: Total Revenue -->
            <div class="col-6 col-lg-3">
                <div class="card stat-card text-bg-success text-white">
                    <div class="card-body">
                        <h6 class="text-white-50 mb-2">Total Revenue</h6>
                        <h3 class="mb-0">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($totalSales, 2) }}</h3>
                        <small class="text-white-50">All income collected</small>
                    </div>
                </div>
            </div>

            <!-- Card 2: Expenses (COGS) -->
            <div class="col-6 col-lg-3">
                <div class="card stat-card text-bg-danger text-white">
                    <div class="card-body">
                        <h6 class="text-white-50 mb-2">Expenses</h6>
                        <h3 class="mb-0">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($totalCOGS, 2) }}</h3>
                        <small class="text-white-50">Stock purchases</small>
                    </div>
                </div>
            </div>

            <!-- Card 3: Sales (Product Sales) -->
            <div class="col-6 col-lg-3">
                <div class="card stat-card text-bg-primary text-white">
                    <div class="card-body">
                        <h6 class="text-white-50 mb-2">Sales</h6>
                        <h3 class="mb-0">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($totalRevenue, 2) }}</h3>
                        <small class="text-white-50">{{ number_format($totalOrders) }} orders</small>
                    </div>
                </div>
            </div>

            <!-- Card 4: Net Profit -->
            <div class="col-6 col-lg-3">
                <div class="card stat-card text-bg-warning text-white">
                    <div class="card-body">
                        <h6 class="text-white-50 mb-2">Net Profit</h6>
                        <h3 class="mb-0 {{ $netProfit >= 0 ? '' : 'text-danger' }}">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($netProfit, 2) }}</h3>
                        <small class="text-white-50">{{ number_format($netProfitMargin, 2) }}% margin</small>
                    </div>
                </div>
            </div>
        </div>

        <div class="row g-4 mb-4">
            <!-- Profit & Loss Statement -->
            <div class="col-12 col-lg-8">
                <div class="card shadow-sm">
                    <div class="card-header bg-body-tertiary">
                        <h5 class="mb-0">Profit & Loss Statement</h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-bordered mb-0">
                                <tbody>
                                    <tr>
                                        <td class="fw-semibold" style="width: 70%;">Total Revenue</td>
                                        <td class="text-end fw-semibold">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($totalSales, 2) }}</td>
                                    </tr>
                                    <tr>
                                        <td class="ps-4 text-secondary">Product Sales</td>
                                        <td class="text-end">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($totalRevenue, 2) }}</td>
                                    </tr>
                                    <tr>
                                        <td class="ps-4 text-secondary">Fuel Charges</td>
                                        <td class="text-end">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($totalFuelCharges, 2) }}</td>
                                    </tr>
                                    <tr>
                                        <td class="ps-4 text-secondary">Other Charges</td>
                                        <td class="text-end">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($totalOtherCharges, 2) }}</td>
                                    </tr>
                                    <tr class="table-secondary">
                                        <td class="fw-semibold">Cost of Goods Sold</td>
                                        <td class="text-end fw-semibold">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($totalCOGS, 2) }}</td>
                                    </tr>
                                    <tr class="table-success">
                                        <td class="fw-semibold">Net Profit</td>
                                        <td class="text-end fw-semibold {{ $netProfit >= 0 ? 'text-success' : 'text-danger' }}">
                                            {{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($netProfit, 2) }}
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Stats -->
            <div class="col-12 col-lg-4">
                <div class="card shadow-sm">
                    <div class="card-header bg-body-tertiary">
                        <h5 class="mb-0">Quick Stats</h5>
                    </div>
                    <div class="card-body">
                        <div class="d-flex justify-content-between mb-3">
                            <span class="text-secondary">Total Orders</span>
                            <strong>{{ number_format($totalOrders) }}</strong>
                        </div>
                        <div class="d-flex justify-content-between mb-3">
                            <span class="text-secondary">Avg Order Value</span>
                            <strong>{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($averageOrderValue, 2) }}</strong>
                        </div>
                        <div class="d-flex justify-content-between mb-3">
                            <span class="text-secondary">Profit Margin</span>
                            <strong class="{{ $netProfitMargin >= 0 ? 'text-success' : 'text-danger' }}">
                                {{ number_format($netProfitMargin, 2) }}%
                            </strong>
                        </div>
                        <hr>
                        <div class="d-flex justify-content-between align-items-center">
                            <span class="fw-semibold">This Month</span>
                            <strong>{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($currentMonthSales, 2) }}</strong>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Ledger -->
        <div class="row">
            <div class="col-12">
                <div class="card shadow-sm">
                    <div class="card-header bg-body-tertiary d-flex justify-content-between align-items-center">
                        <h5 class="mb-0">Ledger</h5>
                        <span class="badge text-bg-primary">{{ $ledger->count() }} transactions</span>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table id="ledgerTable" class="table table-hover mb-0 align-middle table-striped">
                                    <thead class="table-light">
                                        <tr>
                                            <th scope="col" style="width: 100px;">Date</th>
                                            <th scope="col" style="width: 150px;">Type</th>
                                            <th scope="col">Description</th>
                                            <th scope="col" class="text-end" style="width: 120px;">Debit</th>
                                            <th scope="col" class="text-end" style="width: 120px;">Credit</th>
                                            <th scope="col" class="text-end" style="width: 120px;">Balance</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        @if(request('start_date') && request('end_date') && $openingBalance != 0)
                                        <tr class="table-info">
                                            <td>{{ \Carbon\Carbon::parse($startDate)->subDay()->format('d M Y') }}</td>
                                            <td><span class="badge text-bg-secondary">Opening</span></td>
                                            <td>Opening Balance</td>
                                            <td class="text-end">—</td>
                                            <td class="text-end">—</td>
                                            <td class="text-end fw-semibold {{ $openingBalance >= 0 ? 'text-success' : 'text-danger' }}">
                                                {{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($openingBalance, 2) }}
                                            </td>
                                        </tr>
                                        @endif
                                        @forelse($ledger as $entry)
                                        <tr>
                                            <td>{{ \Carbon\Carbon::parse($entry['date'])->format('d M Y') }}</td>
                                            <td>
                                                @if($entry['type'] === 'Sale')
                                                    <span class="badge text-bg-success">Sale</span>
                                                @else
                                                    <span class="badge text-bg-danger">Purchase</span>
                                                @endif
                                            </td>
                                            <td>{{ $entry['description'] }}</td>
                                            <td class="text-end">
                                                @if($entry['debit'] > 0)
                                                    <span class="text-danger">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($entry['debit'], 2) }}</span>
                                                @else
                                                    <span class="text-secondary">—</span>
                                                @endif
                                            </td>
                                            <td class="text-end">
                                                @if($entry['credit'] > 0)
                                                    <span class="text-success">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($entry['credit'], 2) }}</span>
                                                @else
                                                    <span class="text-secondary">—</span>
                                                @endif
                                            </td>
                                            <td class="text-end fw-semibold {{ $entry['balance'] >= 0 ? 'text-success' : 'text-danger' }}">
                                                {{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($entry['balance'], 2) }}
                                            </td>
                                        </tr>
                                        @empty
                                        <tr>
                                            <td colspan="6" class="text-center text-secondary py-4">
                                                <i class="bi bi-inbox me-2"></i>No transactions found for this period
                                            </td>
                                        </tr>
                                        @endforelse
                                    </tbody>
                                    @if($ledger->count() > 0)
                                    <tfoot class="table-light">
                                        <tr>
                                            <td colspan="3" class="fw-semibold">Closing Balance</td>
                                            <td class="text-end fw-semibold text-danger">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($ledger->sum('debit'), 2) }}</td>
                                            <td class="text-end fw-semibold text-success">{{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($ledger->sum('credit'), 2) }}</td>
                                            <td class="text-end fw-semibold {{ ($ledger->last()['balance'] ?? 0) >= 0 ? 'text-success' : 'text-danger' }}">
                                                {{ env('APP_CURRENCY_SYMBOL') }}{{ number_format($ledger->last()['balance'] ?? $openingBalance, 2) }}
                                            </td>
                                        </tr>
                                    </tfoot>
                                    @endif
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
document.addEventListener('DOMContentLoaded', function() {
    const startDateValue = "{{ request('start_date', '') }}";
    const endDateValue = "{{ request('end_date', '') }}";
    
    const startDateConfig = {
        dateFormat: "Y-m-d",
        allowInput: true
    };
    
    const endDateConfig = {
        dateFormat: "Y-m-d",
        allowInput: true
    };
    
    // Only set defaultDate if a value is provided
    if (startDateValue) {
        startDateConfig.defaultDate = startDateValue;
    }
    
    if (endDateValue) {
        endDateConfig.defaultDate = endDateValue;
    }
    
    flatpickr("#start_date", startDateConfig);
    flatpickr("#end_date", endDateConfig);

    // Initialize DataTables for Ledger
    $('#ledgerTable').DataTable({
        responsive: true,
        pageLength: 25,
        lengthMenu: [[10, 25, 50, 100, -1], [10, 25, 50, 100, "All"]],
        order: [[0, 'asc']],
        dom: '<"row"<"col-sm-12"B><"col-sm-12"f>>' +
             '<"row"<"col-sm-12"tr>>' +
             '<"row"<"col-sm-12 col-md-5"i><"col-sm-12 col-md-7"p>>',
        processing: true,
        deferRender: true,
        buttons: [
            {
                extend: 'excel',
                text: '<i class="bi bi-file-earmark-spreadsheet me-1"></i>Excel',
                className: 'btn btn-sm',
                exportOptions: {
                    columns: ':visible'
                }
            },
            {
                extend: 'pdf',
                text: '<i class="bi bi-file-earmark-pdf me-1"></i>PDF',
                className: 'btn btn-sm',
                exportOptions: {
                    columns: ':visible'
                },
                customize: function(doc) {
                    doc.defaultStyle.fontSize = 9;
                    doc.styles.tableHeader.fontSize = 10;
                }
            },
            {
                extend: 'csv',
                text: '<i class="bi bi-filetype-csv me-1"></i>CSV',
                className: 'btn btn-sm',
                exportOptions: {
                    columns: ':visible'
                }
            },
            {
                extend: 'print',
                text: '<i class="bi bi-printer me-1"></i>Print',
                className: 'btn btn-sm',
                exportOptions: {
                    columns: ':visible'
                }
            }
        ],
        language: {
            search: "Search:",
            lengthMenu: "Show _MENU_ entries",
            info: "Showing _START_ to _END_ of _TOTAL_ entries",
            infoEmpty: "Showing 0 to 0 of 0 entries",
            infoFiltered: "(filtered from _MAX_ total entries)",
            paginate: {
                first: "First",
                last: "Last",
                next: "Next",
                previous: "Previous"
            },
            emptyTable: "No data available in table",
            zeroRecords: "No matching records found"
        }
    });
});
</script>
@endpush
@endsection
