<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Foundation\Auth\AuthenticatesUsers;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    use AuthenticatesUsers;

    protected $redirectTo = '/dashboard';

    public function __construct()
    {
        $this->middleware('guest')->except('logout');
        $this->middleware('auth')->only('logout');
    }

    /**
     * Return only email + password
     */
    protected function credentials(Request $request)
    {
        return $request->only($this->username(), 'password');
    }

    /**
     * Custom login validation BEFORE authentication
     */
    protected function attemptLogin(Request $request)
    {
        $user = \App\Models\User::where($this->username(), $request->email)->first();

        if ($user) {

            // 1. User active?
            if ($user->status !== 'active') {
                throw ValidationException::withMessages([
                    $this->username() => ['Your account is inactive. Please contact admin.'],
                ]);
            }

            // 2. Has role?
            $role = $user->roles()->first();
            if (!$role) {
                throw ValidationException::withMessages([
                    $this->username() => ['No role is assigned to your account. Contact admin.'],
                ]);
            }

            // 3. Role exists?
            if (!$role->exists) {
                throw ValidationException::withMessages([
                    $this->username() => ['Your assigned role does not exist anymore. Contact admin.'],
                ]);
            }

            // 4. Role active?
            if ($role->status !== 'active') {
                throw ValidationException::withMessages([
                    $this->username() => ['Your role is inactive. Login not allowed.'],
                ]);
            }
        }

        // If everything okay → attempt authentication
        return $this->guard()->attempt(
            $this->credentials($request),
            $request->filled('remember')
        );
    }

    /**
     * AFTER LOGIN: Redirect based on first available permission
     */
    protected function authenticated(Request $request, $user)
    {
        // Permission -> Route mapping (based on your web.php)
        $permissionRedirectMap = [
            'view-dashboard'     => '/dashboard',
            'view-pos'           => '/create-pos',
            'view-product'       => '/products',
            'view-pack-sizes'    => '/pack-sizes',
            'view-inventory'     => '/inventory',
            'view-stockin'       => '/stock-in-view',
            'view-stockout'      => '/stock-out-view',
            'view-sales'         => '/sales',
            'view-accounting'    => '/accounting',
            'view-users'         => '/users',
            'view-roles'         => '/roles',
            'view-permissions'   => '/permissions',
        ];

        // Check user's permissions in order
        foreach ($permissionRedirectMap as $permission => $path) {
            if ($user->can($permission)) {
                return redirect($path);
            }
        }

        // If user has NO permissions → block login
        auth()->logout();

        return redirect('/login')->withErrors([
            $this->username() => 'You have no permissions assigned. Contact admin.',
        ]);
    }

    /**
     * Default login failed message
     */
    protected function sendFailedLoginResponse(Request $request)
    {
        throw ValidationException::withMessages([
            $this->username() => [trans('auth.failed')],
        ]);
    }
}
